#include "csfunified/featuresets/adapters/MeetingFeatureSetAdapter.h"
#include "csf/logger/CSFLogger.hpp"

#include "boost/bind.hpp"
#include "boost/function.hpp"
#include "jcfcoreutils/StringUtils.h"
#include "csfunified/framework/UnifiedFactory.h"
#include "csfunified/framework/ServicesDispatcher.h"
#include "csfunified/framework/FunctorTask.h"
#include "jcfcoreutils/FunctionTask.h"


#include "services/impl/JabberServiceProvider.h"
#include "services/impl/JabberServiceSink.h"
#include "services/impl/MeetingServiceObjectManager.h"

#ifdef WIN32
#include "sdkwrapper/win32/autoupgrade/jabber_sdk_wrap_impl.h"
#else //MAC
#include "sdkwrapper/mac/MeetingSDKWrap.h"
#include "sdkwrapper/mac/InstantMessageMgr.h"
#include "sdkwrapper/mac/MeetingApiDelegate.h"
#endif

namespace CSFUnified
{
	static CSFLogger* MeetingFeatureSetAdapterLogger = CSFLogger_getLogger("MeetingService-MeetingFeatureSetAdapterLogger");

	MeetingFeatureSetAdapter::MeetingFeatureSetAdapter(UnifiedFactoryPtr unifiedFactory) : unifiedFactory(unifiedFactory)
	{
		m_started = false;
        JabberServiceProvider::getInstance().setUnifiedFactory(unifiedFactory);
        m_pDSController.reset(DesktopSharingControllerImpl::getInstance());
        
        m_pMeetingMgrController.reset(MeetingMgrControllerImpl::getInstance());
        m_pMeetingAccountController.reset(MeetingAccountControllerImpl::getInstance());

        m_cmrMgr.reset(new CMRControllerMgr(unifiedFactory));

        MeetingServiceObjectManager& mgr = MeetingServiceObjectManager::getInstance();
        mgr.setJabberServiceProvider(&JabberServiceProvider::getInstance());
		mgr.setJabberServiceSink(&JabberServiceSink::getInstance());
        mgr.setMeetingMgrController(m_pMeetingMgrController.get());
        mgr.setDesktopSharingController(m_pDSController.get());
        mgr.setMeetingAccountController(m_pMeetingAccountController.get());
        mgr.setCMRMgr(m_cmrMgr.get());

        CSFLogDebugS(MeetingFeatureSetAdapterLogger, "MeetingFeatureSetAdapter::MeetingFeatureSetAdapter()");
	}
    
    MeetingFeatureSetAdapter::~MeetingFeatureSetAdapter()
    {
        m_pDSController.reset();
        m_pMeetingMgrController.reset();
        m_pMeetingAccountController.reset();
    }

	void MeetingFeatureSetAdapter::StartFeatureSet(unsigned int type, ServiceEventVtrPtr errors)
	{
		CSFLogDebugS(MeetingFeatureSetAdapterLogger, "MeetingFeatureSetAdapter::StartFeatureSet");
		m_started = true;
		csf::ScopedReadRWLock lock(mLock);

		SMART_PTR_NS::shared_ptr<ServicesDispatcher> pServiceDispatcher = this->unifiedFactory->getServicesDispatcher();
        if (pServiceDispatcher != NULL)
        {
            bool bOnDispatcher = pServiceDispatcher->checkForUpdateAccess();
            if (bOnDispatcher)
            {
                start();
            }
            else
            {
                pServiceDispatcher->enqueueBlock(boost::bind(&MeetingFeatureSetAdapter::start, this),
                                                 "MeetingFeatureSetAdapter::start");
            }
        }
	}

	void MeetingFeatureSetAdapter::StopFeatureSet(unsigned int type, ServiceEventVtrPtr errors)
	{
		CSFLogDebugS(MeetingFeatureSetAdapterLogger, "MeetingFeatureSetAdapter::StopFeatureSet");
		m_started = false;
		csf::ScopedReadRWLock lock(mLock);

		SMART_PTR_NS::shared_ptr<ServicesDispatcher> pServiceDispatcher = this->unifiedFactory->getServicesDispatcher();
        if (pServiceDispatcher != NULL)
        {
            bool bOnDispatcher = pServiceDispatcher->checkForUpdateAccess();
            if (bOnDispatcher)
            {
                stop();
            }
            else
            {
                pServiceDispatcher->enqueueBlock(boost::bind(&MeetingFeatureSetAdapter::stop, this),
                                                 "MeetingFeatureSetAdapter::stop");
            }
        }
	}

	void MeetingFeatureSetAdapter::Destroy()
	{
		CSFLogDebugS(MeetingFeatureSetAdapterLogger, "MeetingFeatureSetAdapter::Destroy");
		m_started = false;

		//call destroy for each components.
	}

	bool MeetingFeatureSetAdapter::IsFeatureSetStarted(unsigned int type)
	{
		CSFLogDebugS(MeetingFeatureSetAdapterLogger, "MeetingFeatureSetAdapter::IsFeatureSetStarted");
		return m_started;
	}

	bool MeetingFeatureSetAdapter::IsFeatureSetProvisioned(unsigned int type)
	{
		return true;
	}

	std::vector<unsigned int> MeetingFeatureSetAdapter::GetFeatureSetDependencies(unsigned int type)
	{
		std::vector<unsigned int> result;
		return result;
	}

	void MeetingFeatureSetAdapter::start()
	{
		CSFLogDebugS(MeetingFeatureSetAdapterLogger, "MeetingFeatureSetAdapter::start");
        
		JabberServiceProvider::getInstance().start();

#ifdef WIN32
		JM_SDK_WRAP::JabberSDKWrapImpl::getInstance()->startMeetingSDK();
#else
        MeetingApiDelegate::getInstance()->initJabberAccountName();
        MeetingSDKWrap::getInstance()->startMeetingSDK();
        InstantMessageMgr::getInstance()->start();
        JabberServiceProvider::getInstance().startPresenceMonitor();
#endif
        
		//call start for each components.
        m_cmrMgr->start();
        m_pDSController->start();
        m_pMeetingAccountController->start();
        m_pMeetingMgrController->start();
		JabberServiceSink::getInstance().start();
	}

	void MeetingFeatureSetAdapter::stop()
	{
		CSFLogDebugS(MeetingFeatureSetAdapterLogger, "MeetingFeatureSetAdapter::stop");
        
		//call stop for each components
		JabberServiceSink::getInstance().stop();
		m_pDSController->stop();
        m_pMeetingAccountController->stop();
        m_cmrMgr->stop();
        
        m_pMeetingMgrController->stop();
        
#ifdef WIN32
		JM_SDK_WRAP::JabberSDKWrapImpl::getInstance()->stopMeetingSDK();
#else
        JabberServiceProvider::getInstance().stopPresenceMonitor();
        InstantMessageMgr::getInstance()->stop();
        MeetingSDKWrap::getInstance()->stopMeetingSDK();
#endif

		JabberServiceProvider::getInstance().stop();
	}
    
    SMART_PTR_NS::shared_ptr<CSFUnified::MeetingMgrController> MeetingFeatureSetAdapter::getMeetingMgrController()
    {
        return m_pMeetingMgrController;
    }
    
    SMART_PTR_NS::shared_ptr<CSFUnified::DesktopSharingController> MeetingFeatureSetAdapter::getDesktopSharingController()
    {
        return m_pDSController;
    }
    
    SMART_PTR_NS::shared_ptr<CSFUnified::MeetingAccountController> MeetingFeatureSetAdapter::getMeetingAccountController()
    {
        return m_pMeetingAccountController;
    }
    
    SMART_PTR_NS::shared_ptr<CSFUnified::CMRController> MeetingFeatureSetAdapter::getCMRController(std::string JID)
    {
        return m_cmrMgr->getCMRController(JID);
    }
    
    void MeetingFeatureSetAdapter::registerCMRMeetingCallback(SMART_PTR_NS::shared_ptr<CSFUnified::CMRMeetingCallback> callback)
    {
        m_cmrMgr->registerCallback(callback);
    }
    
    void MeetingFeatureSetAdapter::unregisterCMRMeetingCallback(SMART_PTR_NS::shared_ptr<CSFUnified::CMRMeetingCallback> callback)
    {
        m_cmrMgr->unregisterCallback(callback);
    }
    
    
};




